from django.db import models
from django.db.models.signals import pre_save
from django.dispatch import receiver
from django.utils import timezone

# Create your models here.

class OrderInfo(models.Model):
    # Order status choices
    STATUS_CHOICES = [
        ('placed', 'Placed'),
        ('pending', 'Pending'),
        ('confirmed', 'Confirmed'),
        ('delivered', 'Delivered'),
    ]

    order_id = models.IntegerField()
    order_mail = models.CharField(max_length=180)
    order_custom_number = models.IntegerField(null=True, blank=True)
    order_status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='placed',
        null=True,
        blank=True,
    )
    created_at = models.DateTimeField(auto_now_add=True, null=True, blank=True)
    status_updated_at = models.DateTimeField(null=True, blank=True)

    def save(self, *args, **kwargs):
        # Check if order_status has changed
        if self.pk:  # If object already exists
            old_obj = OrderInfo.objects.get(pk=self.pk)
            if old_obj.order_status != self.order_status:
                self.status_updated_at = timezone.now()
        else:  # New object
            self.status_updated_at = timezone.now()
        super().save(*args, **kwargs)

    def __str__(self):
        return str(self.order_id)

class UserData(models.Model):
    orderid = models.ForeignKey(OrderInfo, on_delete=models.CASCADE)
    upload_img = models.ImageField(upload_to='image/%Y/%m/', null=True, default=None, blank=True)
    upload_img2 = models.ImageField(upload_to='image/%Y/%m/', null=True, default=None, blank=True)
    upload_img3 = models.ImageField(upload_to='image/%Y/%m/', null=True, blank=True)
    custom_number = models.IntegerField(unique=True, null=True, blank=True)

@receiver(pre_save, sender=UserData)
def generate_custom_number(sender, instance, **kwargs):
    if not instance.custom_number:  # Check if custom_number is not set
        # Get the last object (if any)
        last_object = UserData.objects.order_by('-custom_number').first()
        new_number = 1 if not last_object else last_object.custom_number + 1
        instance.custom_number = new_number
        